%
%This code accompanies the paper: "Bayesian Image Recovery for Dendritic
%Structures Under Low Signal-to-Noise Conditions"
%
%Running this code displays a slideshow of MCMC samples (samples are 
%separated by  intervals defined in params). MCMC is described in the section 
%of the paper, "Markov Chain Monte Carlo methods for sampling from p(S|I_obs)"
%
%the general order of the code:
% 1. generating simulated data from an experimental neuron image
% 2. blurring then thresholding the data to obtain a starting point for our
% algorithm
%3. executing Gibbs sampling over a space of images which differ by a
%single pixel and displaying a slideshow of resulting MCMC samples
%4. display final figures 
%
%note: Gibbs sampling runs more quickly without the visualization slideshow
%
%this code employs functions from matlab's image processing toolbox and
%statistics toolbox
%
%this code calls the following subfunctions: neighborhood2dec,
%thresholding_function, point_spread_function, and slideshow_figure. these
%functions are explained next to their code, at the end of the paper.
%
%
%Inputs: 
%lambda_i: neuron fluoresce (max fluorescence)
%lambda_o: baseline fluoresce (backround noise)
%maxmarkovcounter: determines number of proposed steps in the MC (not counting steps which break simple connectivity)
%intervalP: number of proposals between samples
%burnin: number of proposals to discard before saving samples
%slideshow_start: number of proposals before starting the slideshow
%alpha1: strength of penalty based on number of edges of edge-set pixels
%alpha2: strength of penalty based on number of elements in the edge set
%final_figures:displays a picture of sampling at each value of alpha1,alpha2
%thresholdNo: number of thresholds to take when searching for S_initial
%startmult: fraction of the mean to start search for best threshold for S_initial
%endmult: fraction of the mean to end search for best threshold for S_initial
%psf_spread: spread  for the gaussian psf which describes blur
%psf_size: determines the diameter of the gaussian-- (diameter=2*psfsize+1)
%
%
%Outputs:
%MarkovProposalPercent: percentage of proposals which were accepted during Gibbs sampling
%S_current: the current shape of the neuron, represented by an n-by-n matrix
%S_crop: cropped S_current, m-by-m matrix with reduced size m= (n - psf_size) to avoid problematic edge areas
 %S_Markov: an array of S_crop, ie the samples from MCMC, taken after burn-in at intervals determined by intervalP
%UnpenalizedLogLikelihood: list of unpenalized loglikelihoods, calculated for S_current after each proposed step
%PenalizedLogLikelihood: similar list for penalized loglikelihood, the difference between the two depends upon penalty strength
%checkedlist: image showing where proposals have occured, along with the number of proposals
%PoissonData: simulated data, generated as described in section: "the blurred poisson image formation model"
%S_true: `true shape' used to generate poisson data, extracted from image as described in appendix
%psf: point spread function used for data generation and later deconvolution, we use a gaussian
%
%code avaliable for edititng and incorporation into other programs 
%contact: Geoff Fudenberg, geoff.fudenberg@gmail.com



function [MarkovProposalPercent,S_current,S_Markov,UnpenalizedLogLikelihood,PenalizedLogLikelihood,checkedlist,PoissonData,S_true,psf,totalperi] = MC_Demo(lambda_i,lambda_o,maxmarkovcounter,...
    intervalP,burnin,alpha1,alpha2,thresholdNo,startmult,endmult,psf_spread,psf_size,slideshow_start)

close all;
load FCvector %library for determining whether flips which add to the neuron change connectivity   
load UCvector %library for determining whether removals from the neuron change connectivity
alpha1o=alpha1; alpha2o=alpha2;
stepsize= (endmult-startmult)/thresholdNo;





%============== creating simulated data, following the paper's discussion of "data generation" ====================
%1. isolate neuronal shape
%2. generate poisson data according to : Imageout= Poiss{Imagein*psf+noise}
disp('generating simulated data')


%start by scaling the image to a chosen size & setting up a coordinate system
dx= .1; %this defines the coarseness of the grid
neuronal_image= imread( 'neuron.tif'); %actual neuron data from a TPLSCM
neuronal_image = neuronal_image(51:551,1:501);
Id = imresize(neuronal_image,.04/dx); 
dy=dx; x= -10+dx/2:dx:10-dx/2; y= -10+dy/2:dy:10-dy/2; halfx= round(length(x)/2); halfy= round(length(y)/2);
xmax=length(x); ymax=length(y);
Id = single(Id);
Id = Id(1:2*halfx,1:2*halfy);
[rows columns]= size(Id);
nextdoor= [-rows-1 -rows -rows+1 -1 1 rows-1 rows rows+1];

%we create a gaussian function to smooth out the imported image; we can then extract a neuronal shape for data generation
 N=15; nspan=-N:N;  spread=.25;   psf_dx= .1;
psfd = point_spread_function(N,psf_dx,spread,nspan); 

%smoothing the tif data
Idsmooth = convolve2(Id,psfd,'same');
thresholdvalue = 2.5*mean(mean(Idsmooth));
thresholddata = (Idsmooth >= thresholdvalue) ;
%the bwlabel function numbers the connected regions
cthresholddata= bwlabel(thresholddata,4);
%regionprops with the 'Area' string counts the components of each numbered region
%we take the largest connected region as our extracted shape
arealist = regionprops(cthresholddata,'Area');
areavector = zeros(1,max(max(cthresholddata)));
    for i = 1:max(max(cthresholddata))

             areavector(1,i) = arealist(i).Area;
    end
[maxarea,largestregion] = max(areavector);
largestdata= cthresholddata==largestregion;


%===============data generation: poisson data ==============
S_true = imfill(largestdata,'holes'); % this corresponds to the simply connected initial neuronal shape

%we now define the psf for actual use: this psf is based upon physical
%parameters of psf size: ie. it uses the s=2*.04
N= psf_size; % this sets the size of the gaussian elements that aren't ignored
nspan=-N:N;    %define here to speed up some code
spread=psf_spread;
psf= point_spread_function(psf_size,dx,spread,nspan);

%described in section: the blurred Poisson image formation model
%we set baseline (lambda_o) and internal (lambda_i) fluorescence levels for the shape, and convolve with the psf
S_true(:,1:N)=0; S_true(:,end-N:end)=0; S_true(1:N,:)=0; S_true(end-N:end,:)=0;
S_compare= S_true(N:end-N,N:end-N);
In = (lambda_i-lambda_o) .*S_true;
Iconv = convolve2(In,psf,'same');
lambda_true = Iconv + lambda_o;


%to create the random data: we draw from the distribution with poisson
%count rate at each pixel given by lambda_true multiplied by the bin size (dx*dy);
PoissonData = poissrnd(lambda_true*dx*dy);

%note:::: in the code that follows, we will denote lambda_current, or the
%convolution of the shape which takes fluorescence into account, as Is


%======choosing the threshold: finding an optimal initial guess for neuron shape==========
%described in  section:  Appendix: Initialization of S

disp('thresholding data for initial shape')
%to smear out the data
PoissonDataSmooth = convolve2(PoissonData,psfd,'same');


PoissonData= sparse(PoissonData); 
AllDataPoints=find(PoissonData);
loglikelihoods = zeros(1,thresholdNo);

%we take thresholds according to the section "thresholding/functional form
%of L" in the paper
%to achieve this we input the smoothed data and to produce a starting shape
%(ie. a 2D step-function)
%for the starting shape we take the most likely thresholded shape, given
%the data the range over which we search (defined @ beginning)
[S_current] = thresholding_function(thresholdNo,PoissonDataSmooth,PoissonData,AllDataPoints,psf,lambda_i,lambda_o,dx,dy,startmult,stepsize);


%throw out edges to avoid convolution effects etc.
S_current(:,1:N)=0; 
S_current(:,end-N:end)=0;

%this is the initial guess for neuron shape
S_initial = S_current;



%==== calculating lamba^(k)=============
% we find lambda^k because it is used in turn to find the likelihood
%of the current shape given the data
Is= S_current*(lambda_i-lambda_o);
Is= convolve2(Is,psf,'same')+lambda_o;
%we only need to keep track of those points were data entries are non-zero,
%and thus store in sparse format
IsSp=zeros(size(Is));
IsSp(find(PoissonData))=Is(find(PoissonData));
Is=IsSp;
Is=sparse(Is);
IsInitial=Is;


%======single pixel convolved with psf============================
%we now find the perturbative function, or a single pixel convolved with
%the psf. by doing so we avoid needing to reconvolve the whole step function
%after each step of MCMC. instead we can simply update the altered region
%additively: see section "iterative algorithm" for details on how this time-saving 
%step depends upon the linearity of convolutions 
pI= zeros(length(x),length(x));
    halfx= round(length(x)/2);
    halfy= round(length(y)/2);
pI(halfx,halfy)=lambda_i-lambda_o;
pspread=convolve2(pI,psf,'same');
pIs=pspread;
pIsc=pIs(halfx+nspan,halfy+nspan);
psum= sum(sum(pIs))*dx*dy;




%=======to find the perimeter given a thresholded step function==========
%see section "edge set" of the paper
S_current= S_current ;
S_left= zeros(size(S_current));
S_left(:,1:length(S_current)-1)= S_current(:,2:length(S_current));
S_right= zeros(size(S_current));
S_right(:,2:length(S_current))= S_current(:,1:length(S_current)-1);
S_up= zeros(size(S_current));
S_up(1:length(S_current)-1,:)= S_current(2:length(S_current), :);
S_down= zeros(size (S_current));
S_down(2:length(S_current),:)= S_current(1:length(S_current)-1,:);

totalperi= (4*S_current-S_left-S_right -S_up - S_down) >0;

totalperiS=totalperi; % for debugging

ftp=find(totalperi); 



%===== to find the outer neighbors===============
%see section "boundary set" and "simple connectivity constraint" for context
%for MCMC we must keep track of external neighbors in order to sample randomly 
%from the set of all simply connected images differing by one pixel from S_current

outerperi=zeros(size(totalperi)); 
for i=1:length(ftp); 
    px=ftp(i); 
    if ~S_current(px+rows); 
        outerperi(px+rows)=1;
    end
    if  ~S_current(px+1); 
        outerperi(px+1)=1;   
    end 
    if   ~S_current(px-1); 
        outerperi(px-1)=1;   
    end
     if   ~S_current(px-rows); 
        outerperi(px-rows)=1;  
    end
end

outerperiInitial= outerperi;

%========    the loglikelihood for a poisson process ===========
AllDataPoints= find(PoissonData);
%calculating initial log likelihood as described in section: "computing the likelihood"
loglikelihoodcandidate= -sum(sum(Is))*dx*dy+sum(sum(PoissonData(AllDataPoints).*log(Is(AllDataPoints)))); 
old_likelihood=loglikelihoodcandidate;


%====== creating the list of edge set and external neighbors======
%proposals for MCMC are generated by choosing a random element of
%ProposalList which includes the list of edge set elements and outer
%neighbors
perilist= find(totalperi);
perilistS=perilist;
outerperilist=find(outerperi);
outerperilistS=outerperilist;
ProposalList=[perilist;outerperilist]; % propose perimeter elements from this for MC sample update

%=== calculating alpha1 penalty===========================
% all that is necessary is simply finding the number of the perimeter
% elements (ie. the length of the list)
lperi=length(perilist);
outerlperi=length(outerperilist);
ProposalCardinality=length(ProposalList);
minperilist= 1;

%========calculating the alpha2 penalty===================
%as the alpha2 penalty is based upon the number of edges to edge set pixels
%(basically the perimeter), we can find this by simply adding up the
%outside neighbors for each pixel
for l=1:lperi
    perivalue(l)= sum(~S_current([ (perilist(l)-rows) (perilist(l)-1) (perilist(l)+1) (perilist(l)+rows)]));
end
sumperivalue=sum(perivalue);
sumperivalueS=sumperivalue;

% saving initial perimeter data
        lperiS=lperi;
        outerlperiS=outerlperi;
        ProposalCardinalityInitial=ProposalCardinality;

%==== for storing/recalling/calculating with data from MH sampling=================

%to keep track of edge elements flipped in or out
fliplikelihood= zeros(1,2);
checkedlist= zeros(length(S_current),length(S_current)); 


%sparse arrays for storing information on the number of bins with data
        %these sparse matrices store a list of the nonzero elements of the poisson data around a given pixel due to the form of the loglikelihood, 
        %we only need to calculate the sum for pixels with values larger than 0; also, if a change to the pixel has been proposed earlier, saving the
        %local neighborhood with nonzero values speeds subsequent proposals, either into or out of the neuron, for the pixel
        %finally, sparse matrices are ideal, because proposals durning Gibbs sampling are for only pixels near the perimeter
LocalDataPixels= sparse(length(S_current)^2,(2*N+1)^2,6000*2*N^2);
GlobalDataPixels= sparse(length(S_current)^2,(2*N+1)^2,6000*2*N^2);
LocalData=zeros(size(S_current));
LocalData=sparse(LocalData);

%to store the number of pixels correct at a given sample, based upon comparison with initial shape (S_initial)      
PixelPercentMarkov=zeros(1,round((maxmarkovcounter-burnin)/intervalP));

%to store images of samples from the Markov Chain
S_Markov= zeros(length(S_compare), length(S_compare),round((maxmarkovcounter-burnin)/intervalP));

%to store likelihood graphs
UnpenalizedLogLikelihood= zeros(1,maxmarkovcounter); %the unpenalized likelihood
PenalizedLogLikelihood= zeros(1,maxmarkovcounter);  %the penalized likelihood


    
    disp('beginning MCMC')
    
    %to repeat MC sampling w/different penalty values etc., we need to reset the
    %following variables to their initial values:
    S_current= S_initial;    Is=IsInitial;
    lperi=lperiS;    outerlperi=outerlperiS;
    outerperi=outerperiInitial;    totalperi=totalperiS;    
    perilist=perilistS;    outerperilist=outerperilistS;
    ProposalList= [ perilist; outerperilist];
    ProposalCardinality=lperi+outerlperi;
    sumperivalue=sumperivalueS;

    old_likelihood=loglikelihoodcandidate;
    fliplikelihood(2)= loglikelihoodcandidate-alpha1*sumperivalue-alpha2*lperi;

    markovflips=0; contcount=0; markovcounterNZ=1;  intervalcount=0; intervalcountP=0;
    UnpenalizedLogLikelihood(markovcounterNZ)= old_likelihood;
    PenalizedLogLikelihood(markovcounterNZ)= fliplikelihood(2);


    S_cropped=S_current(N:end-N,N:end-N);
    [picture] = slideshow_figure(S_true(N:end-N,N:end-N),PoissonData(N:end-N,N:end-N),S_initial(N:end-N,N:end-N),S_cropped,markovcounterNZ,maxmarkovcounter);
                    

    
    

%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~markov chain~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%    
    
while markovcounterNZ < maxmarkovcounter

    if markovcounterNZ == burnin
        disp('burnin over')
    end
    
    %simply choose random elements of the perimeter + outer neighbors
    maxperilist= ProposalCardinality;
    perichoice = ProposalList(round(minperilist + (maxperilist-minperilist) * rand(1)));
    xchoice= rem(perichoice-1,rows)+1;
    ychoice= (perichoice-xchoice)/rows+1;
    

%we reject choices too close to the edge (based upon the size 
%of psf) to avoid sampling near edges to bypass
%consequent edge-related problems related to convolution
            if xchoice < 2*N + 1 
                continue
            end
            if xchoice > xmax-2*N -1

                continue
            end
            if ychoice < 2*N +1 
                continue
            end
            if ychoice > ymax-2*N -1
                continue
            end    

%to speed up computation, xNN & yNN label the area around the chosen pixel
%which is necessary to consider for local loglikelihood updating (ie. the size of the psf)
 xNN= xchoice+nspan;
 yNN= ychoice+nspan;
 
%to determine simple connectivity, we extract the 3x3 area around the pixel
%so we can examine it in the context of the section "simple connectivity" 
Ichosen(1:3,1:3)=S_current(xchoice-1:xchoice+1,ychoice-1:ychoice+1);
fliplikelihood(1)= 0;


%==============================checking proposals===================================
%=============2 cases: removal of edge set element from neuron or addition of outside neighbor===========

%===============case 1: removing an edge set element- =====================
%this is the relevant case if the chosen perimeter element falls inside the neuron
if S_current(perichoice) 

    %unflip center pixel
    aIchosen=Ichosen;
    aIchosen(5)= 0;
    %==================check simple connectivity====================
    %to check simple connectivity we first convert the 3x3 neighborhood of S_current around the chosen pixel
    %into a binary number, and then reference the proposal with a list of accepted changes to 3x3 areas to 
    % determine whether there are any connectivity changes
    [dec] =  neighborhood2dec(Ichosen); 
    isdiff= ~UCV(dec); 

    if ~isdiff 
         %if the proposal does not break simple connectivity; we
         %calculate likelihood and update the shape probabilistically
        markovcounterNZ=markovcounterNZ +1;
        %========checks whether the pixel has been proposed before; if not, we update library of proposed areas===============
        %this checks whether the pixel has been proposed before, if it
        %hasn't, we need to input data into the sparse matrix listing
        %which of its neighbors have nonzero data values
                        if LocalData(perichoice) == 0

                            LocalDataPoints= find(PoissonData(xchoice+nspan,ychoice+nspan));
                            LDataPoints=length(LocalDataPoints);
                            LocalData(perichoice)= LDataPoints;
                            LocalDataPixels(perichoice,1:LDataPoints)=LocalDataPoints;
                            localrow= rem(LocalDataPoints-1,length(nspan))+1;
                            localcol= (LocalDataPoints-localrow)/length(nspan)+1;
                            globalrow=localrow-(N+1)+xchoice;
                            globalcol=localcol-(N+1)+ychoice;
                            GlobalDataPoints=globalrow+rows*(globalcol-1);
                            GlobalDataPixels(perichoice,1:LDataPoints)=GlobalDataPoints;

                        end

        LData= LocalData(perichoice);
        LocalIndDataPixels=LocalDataPixels(perichoice,1:LData);%locally indexed data, necessary for computing with pIsc
        GlobalIndDataPixels= GlobalDataPixels(perichoice,1:LData); % globally indexed data, necessary for computing with Iunflip, Is, Ip

        %calculate the new perimeter (necessary for finding perimeter penalty)
        Itunflip= S_current;

        %=========calculate alpha1 penalty =================
        %boutsum, atouchsum & sumdiff are used to update the a1 penalty
        boutsum=sum(~Itunflip([perichoice-rows perichoice-1 perichoice+1 perichoice+rows])); %number of outside neighbors touching the chosen edge pixel
        Itunflip(perichoice)= 0; %updating the 3x3 area extracted from S to reflect the removed pixel 
        atouchsum=sum(Itunflip([perichoice-rows perichoice-1 perichoice+1 perichoice+rows])); %number of inside pixels touching unflipped chosen edge pixel
        sumdiff= boutsum-atouchsum; %the change in outside neighbors
        sumperivalueunflip= sumperivalue - sumdiff; %updating the number of outside neighbors, which is weighted by alpha1 to determine first penalty
        
    
        %===========find perimeter if chosen edge pixel is removed================
            op=totalperi([ perichoice-rows perichoice-1 perichoice+1 perichoice+rows]); 
            totalperiunflip= totalperi;
            totalperiunflip([ perichoice-rows perichoice-1 perichoice perichoice+1 perichoice+rows])= aIchosen([2 4 5 6 8]);
            %eliminate the flipped guys from the perimeter list
            perilistunflip= perilist;
            perilistunflip(perilistunflip==perichoice)= []; %the edge pixel flipped out is never a part of the updated perimeter
            %add in the new neighbors: the new elements are not in the old perimeter
            %and touch the removed pixel, since they have changed from
            %non-perimeter elements to elements of the perimeter
            newElements= and(~op, totalperiunflip([ perichoice-rows perichoice-1 perichoice+1 perichoice+rows])); 
            newElements= newElements.* [ perichoice-rows perichoice-1 perichoice+1 perichoice+rows];
            newElements=newElements(find(newElements));
            perilistunflip= [perilistunflip; newElements(:)]; %updated perimeter element list
            lperiunflip=length(perilistunflip); %updating number of perimeter elements, which is weighted by alpha2 to determine second penalty           

   
         %=========find the outer perimeter if chosen edge pixel removed ============ 
            outerperiunflip=outerperi;
            outerperiunflip(perichoice)=1; %the removed pixel always becomes an outer neighbor
            outerperilistunflip=outerperilist;
            outerperilistunflip(end+1)= perichoice;
            n4= [-rows -1 1 rows];
            for i= 1:4
                ochoice= perichoice+n4(i);
                if ~Itunflip(ochoice)
                    if outerperiunflip(ochoice) == 1
                        if all(Itunflip(ochoice+n4)==0)
                            outerperiunflip(ochoice)=0;
                            outerperilistunflip(outerperilistunflip==ochoice)=[];
                        end
                    end
                end
            end
                        
       
    %==============update the convolved function & calculate loglikelihood===========
    % use the linearity of convolutions discussed in section "iterative algorithm" to avoid reconvolution
    Iunflip= Is;
    localbefore= sum(sum(PoissonData(GlobalIndDataPixels).*log(Is(GlobalIndDataPixels))));
    Iunflip(GlobalIndDataPixels)=Is(GlobalIndDataPixels)-pIsc(LocalIndDataPixels); %updated smoothed function 
    localafter= sum(sum(PoissonData(GlobalIndDataPixels).*log(Iunflip(GlobalIndDataPixels))));
    localsum= psum;
    unflip= old_likelihood + localafter - localbefore + localsum; %raw likelihood
    fliplikelihood(1)= unflip-alpha1*sumperivalueunflip-lperiunflip*alpha2; %calculate the log likelihood of the proposal
    checkedlist(xchoice,ychoice)= checkedlist(xchoice,ychoice)+1;

    
    %==========now we decide probabilistically whether we should accept proposal======
    %see section "markov chain monte carlo methods for sampling"
    pfunct= exp( -(fliplikelihood(2)-fliplikelihood(1)));

        if rand < pfunct
            %update variables if the proposal is accepted (ie I^(k+1)  ~=  I^(k) )
            markovflips= markovflips+1;
            Is= Iunflip; %update the smoothed function (basically the lamba^(k) associated with S_current)
            S_current= Itunflip; %update the thresholded function (shape)
            totalperi= totalperiunflip; %update the perimeter matrix representation
            perilist= perilistunflip; %update the perimeter list representation
            outerperi=outerperiunflip;
            outerperilist=outerperilistunflip;
            ProposalList= [ perilist; outerperilist];
             lperi=lperiunflip; % updating the length of the perimeter
            outerlperi=length(outerperilist);
             ProposalCardinality= outerlperi+lperi;
            old_likelihood=unflip; %update the unpenalized likelihood
            sumperivalue=sumperivalueunflip; %update the a1 function
            fliplikelihood(2)= fliplikelihood(1); %update the penalized likelihood for comparison

            
            
        end    
            UnpenalizedLogLikelihood(markovcounterNZ)= old_likelihood;
            PenalizedLogLikelihood(markovcounterNZ)= fliplikelihood(2);

            
     %========= update slideshow at intervals with spacing intervalP, and start saving samples after the burnin period is over==========
                 if markovcounterNZ > slideshow_start
                     if rem(markovcounterNZ,intervalP) == 0
                         S_crop=S_current(N:end-N,N:end-N);
                         [picture] =  slideshow_figure(S_true(N:end-N,N:end-N),PoissonData(N:end-N,N:end-N),S_initial(N:end-N,N:end-N),S_crop,markovcounterNZ,maxmarkovcounter);
                         if markovcounterNZ > burnin
                             intervalcountP=intervalcountP+1;
                             samepixelsend= (S_compare == S_crop);
                             PixelPercentMarkov(intervalcountP)= sum(samepixelsend(:))/(length(samepixelsend)^2);
                             S_Markov(:,:,intervalcountP)=S_crop;
                         end
                     end
                 end
            
    end

%===============case 2: adding an outside neighbor =====================
%this is the relevant case if the chosen perimeter element falls outside the neuron
elseif ~S_current(perichoice) 
    
    
    %flip pixel at perichoice pixel(add it)
    aIchosen=Ichosen;
    aIchosen(5)= 1;
    %==================check simple connectivity====================
    [dec] =  neighborhood2dec(Ichosen);
    isdiff= ~FCV(dec);
    

    if ~isdiff % if this does not break simple connectivity, then we decide whether to update probabilistically
         markovcounterNZ=markovcounterNZ +1;


            %============== check whether the pixel has been proposed before; if not, update library==========
                if LocalData(perichoice) == 0

                    LocalDataPoints= find(PoissonData(xchoice+nspan,ychoice-1+nspan));

                    LDataPoints=length(LocalDataPoints);
                    LocalData(perichoice)= LDataPoints;
                    LocalDataPixels(perichoice,1:LDataPoints)=LocalDataPoints;
                        localrow= rem(LocalDataPoints-1,length(nspan))+1;
                        localcol= (LocalDataPoints-localrow)/length(nspan)+1;
                        globalrow=localrow-(N+1)+xchoice;
                        globalcol=localcol-(N+1)+ychoice-1;
                    GlobalDataPoints=globalrow+rows*(globalcol-1);
                    GlobalDataPixels(perichoice,1:LDataPoints)=GlobalDataPoints;

                end
            LData= LocalData(perichoice);
            LocalIndDataPixels=LocalDataPixels(perichoice,1:LData);%locally indexed data, necessary for pIsc
            GlobalIndDataPixels= GlobalDataPixels(perichoice,1:LData);     % globally indexed data, necessary for Iunflip, Is, Ip
        
            %============calculate alpha1 penalty========================
            It2= S_current;
            btouchsum=sum(It2([perichoice-rows perichoice-1 perichoice+1 perichoice+rows ])); %the number of inside neighbors touching the empty cell 
            %update the perimeter
            totalperi2= totalperi;
            perilist2= perilist;
            It2(perichoice)= 1; 
            aoutsum=sum(~It2([perichoice-rows perichoice-1 perichoice+1 perichoice+rows ])); %the number of outside neighbors touching the cell after flipping inside
            sumdiff= btouchsum-aoutsum;
            sumperivalue2= sumperivalue - sumdiff;
            
            %===================calculate alpha2 penalty & perimeter if we add chosen pixel=================
            if any(~(It2([perichoice-rows perichoice-1 perichoice+1 perichoice+rows ]))) == 1;
                totalperi2(perichoice)= 1 ;
                perilist2(end+1)= perichoice;
            end
            insideNeighbors= (It2([perichoice-rows perichoice-1 perichoice+1 perichoice+rows ]).* [perichoice-rows perichoice-1 perichoice+1 perichoice+rows]);
            insideNeighbors=insideNeighbors(find(insideNeighbors));
            insideNeighborsNeighbors=[insideNeighbors-rows ;insideNeighbors-1; insideNeighbors+1 ;insideNeighbors+rows];
            removefromperi= insideNeighbors(find(all(It2(insideNeighborsNeighbors))));
            totalperi2(removefromperi)= 0;
            for i=1:length(removefromperi)
                perilist2(perilist2==removefromperi(i))= [];
            end
            lperi2= length(perilist2);
            
            
            %=============calculate outer perimeter if we add chosen pixel===============
            outerperiflip=outerperi;
            outerperiflip(perichoice)=0;
            outerperilistflip=outerperilist;
            outerperilistflip(outerperilistflip==perichoice)= []; %the edge pixel flipped out is never a part of the updated perimeter
            for i=1:4;
                n4=[ -rows -1 1 rows];
                ochoice=perichoice+n4(i);
                if ~It2(ochoice)
                    if any(It2(ochoice+n4))
                        outerperiflip(ochoice)=1;
                        outerperilistflip(end+1)=ochoice;
                    end
                end
            end

            %==============update the convolved function & calculate loglikelihood===========
            Iflip2 = Is;
            localbefore= sum(sum(PoissonData(GlobalIndDataPixels).*log(Is(GlobalIndDataPixels))));
            Iflip2(GlobalIndDataPixels)=Is(GlobalIndDataPixels)+ pIsc(LocalIndDataPixels);
            %the ranges of x,y, and the sign in front of pIsc are different because we are adding  instead of removing a pixel
            localafter= sum(sum(PoissonData(GlobalIndDataPixels).*log(Iflip2(GlobalIndDataPixels))));
            localsum= -psum;
            flip2 = old_likelihood + localafter - localbefore + localsum;
            fliplikelihood(1)=     flip2  - alpha1*sumperivalue2-lperi2*alpha2;

            checkedlist(xchoice,ychoice)= checkedlist(xchoice,ychoice)+1;


    %==========now we again decide probabilistically whether to accept proposal======
            pfunct= exp( -(fliplikelihood(2)-fliplikelihood(1)));

                            if rand < pfunct
                                %update variables if the flip is accepted
                                 markovflips= markovflips+1;
                                 Is = Iflip2;
                                 S_current= It2;
                                 totalperi= totalperi2;
                                 perilist= perilist2;
                                 outerperi=outerperiflip;
                                 outerperilist= outerperilistflip;
                                 outerlperi=length(outerperilistflip);
                                 ProposalList= [ perilist; outerperilist];
                                 lperi=lperi2;
                                 ProposalCardinality=lperi2+outerlperi;
                                 old_likelihood=flip2;
                                 sumperivalue=sumperivalue2;
                                 fliplikelihood(2)= fliplikelihood(1);



                                end
                    UnpenalizedLogLikelihood(markovcounterNZ)= old_likelihood;
                    PenalizedLogLikelihood(markovcounterNZ)= fliplikelihood(2);

                   %========= update slideshow at intervals with spacing intervalP, and start saving samples after the burn-in period is over==========
                   if markovcounterNZ > slideshow_start
                       if rem(markovcounterNZ,intervalP) == 0
                           S_crop=S_current(N:end-N,N:end-N);
                           [picture] =  slideshow_figure(S_true(N:end-N,N:end-N),PoissonData(N:end-N,N:end-N),S_initial(N:end-N,N:end-N),S_crop,markovcounterNZ,maxmarkovcounter);
                           if markovcounterNZ > burnin
                               intervalcountP=intervalcountP+1;
                               samepixelsend= (S_compare == S_crop);
                               PixelPercentMarkov(intervalcountP)= sum(samepixelsend(:))/(length(samepixelsend)^2);
                               S_Markov(:,:,intervalcountP)=S_crop;
                           end
                       end
                   end

    end
         
         
       
    
end %end if statement about whether in or out


     

end %the markov flipping
disp('MCMC over, final figures')

%percentage of the proposals accepted:
MarkovProposalPercent= markovflips/maxmarkovcounter;





%----------------------------------final figures-----------------------------  
%these final figures are meant to display the degree to which the
%loglikelihood converges and to give a sense of the variability in the
%samples chosen by MCMC. Top left: displays the unpenalized loglikelihood
%as a function of proposal number, overlaid with markers showing when the
%burnin ends (ie. start to save MCMC sampels), and the proposal number where the
%the two different samples were chosen. Top Right: displays a superposition
%of the MCMC samples to give an idea of how they vary around the true
%shape. this figure is analagous to the 3rd column in figures 6 & 7 in the paper.
%Bottom Left & Right: display 2 different MCMC samples superposed with the 
%true shape, pixels in red & dark blue are correct, those in light blue are false
%positives, and yellow pixels denote a miss


figure(2)
ysubs=2;
xsubs=2;
sub=0;

%to choose 2 random samples after the burn-in to display
samples=floor((maxmarkovcounter-burnin)/intervalP);
sample1= ceil(samples/2*rand);
sample2= floor(samples/2*rand)+round(samples/2);
sampleNum1=burnin+intervalP*sample1;
sampleNum2=burnin+intervalP*sample2;

sub=sub+1; subplot(ysubs,xsubs,sub);
%display the convergence of the likelihood
plot(UnpenalizedLogLikelihood(:)); title('unpenalized loglikelihood'); xlabel('proposals')
hold on; line(burnin,min(UnpenalizedLogLikelihood(:)):max(UnpenalizedLogLikelihood(:)),'Marker','o','MarkerSize',3);
text(burnin, min(UnpenalizedLogLikelihood(:)),'end of burnin','EdgeColor','black','FontSize',14)
hold on; line(sampleNum1,min(UnpenalizedLogLikelihood(:)):max(UnpenalizedLogLikelihood(:)),'Marker','o','MarkerSize',3);
text(sampleNum1, min(UnpenalizedLogLikelihood(:)) + 1/4*(-min(UnpenalizedLogLikelihood(:))+max(UnpenalizedLogLikelihood(:))),'sample1','EdgeColor','black','FontSize',14)
hold on; line(sampleNum2,min(UnpenalizedLogLikelihood(:)):max(UnpenalizedLogLikelihood(:)),'Marker','o','MarkerSize',3);
text(sampleNum2,  min(UnpenalizedLogLikelihood(:)),'sample2','EdgeColor','black','FontSize',14)



sub=sub+1; subplot(ysubs,xsubs,sub);
%display the superposition of markov samples:
imagesc(sum(S_Markov,3)/max(S_Markov(:))); colorbar; title('superposition of samples'); axis off;
sub=sub+1; subplot(ysubs,xsubs,sub);


%to display the 2 random samples from the distribution & compare them with the true shape
imagesc(S_Markov(:,:,sample1)+2*S_compare); title(['sample1: proposal ', num2str(sampleNum1), ' of  ', num2str(maxmarkovcounter) ]); axis off;
sub=sub+1; subplot(ysubs,xsubs,sub); 
imagesc(S_Markov(:,:,sample2)+2*S_compare); title(['sample2: proposal ', num2str(sampleNum2), ' of  ', num2str(maxmarkovcounter) ]); xlabel('red & dark blue = correct, light blue = false positive, yellow = miss'); axis off;
text(-250,220,'red & dark blue = correct, light blue = false positive, yellow = miss');



checkedlist= checkedlist(N:end-N,N:end-N,:,:,:);


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%subfunctions

%code which displays the picture comparing the current shape of our
%proposed neuron with the true shape, the inital shape, and the underlying
%data
function [picture] = slideshow_figure(S_true,PoissonData,S_initial,S_crop,markovcounterNZ,maxmarkovcounter)
            figure(1)
            ysubs=2;
            xsubs=2;
            sub=0;

            sub=sub+1; subplot(ysubs,xsubs,sub);
            imagesc(S_true); title('true shape'); axis off;
            sub=sub+1; subplot(ysubs,xsubs,sub);
            imagesc(PoissonData); colorbar; title('poisson data'); axis off;
            sub=sub+1; subplot(ysubs,xsubs,sub);
            imagesc(S_initial); title('initial shape'); axis off;
            sub=sub+1; subplot(ysubs,xsubs,sub); 
            imagesc(S_crop);  title(['current shape, MCMC iteration ', num2str(markovcounterNZ) ,' of ', num2str(maxmarkovcounter)]);  axis off;  
            picture=1;

%code which creates a gaussian function, we use a gaussian psf to
%blur the data according to our image degradation model
function [psf] = point_spread_function(psf_size,dx,spread,nspan)
            xaug= -2*psf_size*dx:dx:2*psf_size*dx; yaug= xaug;
            [xx,yy]=meshgrid(xaug,yaug);
            halfpsfx=round(length(xaug)/2); halfpsfy= round(length(yaug)/2);
            psf= exp(-.5*(xx.^2+yy.^2)/spread^2)/(2*pi*spread^2);
            psf(psf<eps*max(psf(:))) = 0;
            psfN=psf(halfpsfx+nspan,halfpsfy+nspan);
            psfN=psfN/sum(psfN(:));
            psf=psfN;

%code for taking thresholds of the blurred/smoothed poisson data, we take thresholds as
%functions of the mean of the smoothed data and choose most likely upon
%computation of poisson likelihood
function [S_current] = thresholding_function(thresholdNo,PoissonDataSmooth,PoissonData,AllDataPoints,psf,lambda_i,lambda_o,dx,dy,startmult,stepsize)

        for k = 1:thresholdNo


            j = startmult+stepsize*k;
            thresholdvalue = j*mean(mean(PoissonDataSmooth));

            thresholddata = (PoissonDataSmooth >= thresholdvalue) ;

            %the bwlabel function numbers the connected regions
            cthresholddata= bwlabel(thresholddata,4);

            %---> do this better w/summation: regionprops with the 'Area' string counts the components of each numbered region
            arealist = regionprops(cthresholddata,'Area');

            areavector = zeros(1,max(max(cthresholddata)));

            % ---> there should be a better way to do this involving summation and ==
            for i = 1:max(max(cthresholddata))

                areavector(1,i) = arealist(i).Area;

            end

            [maxarea,largestregion] = max(areavector);

            %this gives the largest connected region
            %largestregion;
            %size(cthresholddata);
            %size(largestregion);
            if min(size(largestregion)) == 1
                largestdata= cthresholddata==largestregion;

            else
                maxthreshold = j
                continue
            end

            %----> should be a way to do this better: to fill in points that are completely surrounded
            largestdatafilled = imfill(largestdata,'holes');

            %first add baseline fluoresce to threshold function, then convolve with
            %psf

            candidate = (lambda_i-lambda_o) .*largestdatafilled;

            candidatesmooth= convolve2(candidate,psf,'same');

            candidatephys = candidatesmooth + lambda_o;


            loglikelihood= -sum(sum(candidatephys))*dx*dy+ sum( sum(PoissonData(AllDataPoints).*log(candidatephys(AllDataPoints))));

            loglikelihoods(1,k) = loglikelihood;


            if loglikelihoods(1,k) == max(loglikelihoods(find(loglikelihoods)))
                bestcandidate= candidatephys;
                bl = loglikelihood;
                bc=k;
                bestthreshold = j;
                S_current= single(largestdatafilled);
            end
        end

        
%code for translating a 3x3 neighborhood into a natural number; this is used as natural numbers form
% the indices for the connectivity vectors for both adding and removing a pixel. the connectivity library 
%that these vectors constitute are explained in "simple connectivity constraint"
function [dec] =  neighborhood2dec(Ichosen)

            BB= (Ichosen(:))';
            CC= setstr(BB+48);
            [MM,NN] = size(CC);
            % Convert to numbers
            VV = CC - '0'; 
            twos = pow2(NN-1:-1:0);
            XX = sum(VV .* twos(ones(MM,1),:),2);
            dec= XX +1;      
        
